// CleanFinding Extension - Content Script
// Blocks inappropriate content and trackers on all websites

(function() {
  'use strict';

  // Get settings
  let settings = {
    blockTrackers: true,
    blockInappropriate: true,
    blockAnalytics: true,
    enforceSafeSearch: true
  };

  // Load settings from storage
  chrome.runtime.sendMessage({ action: 'getSettings' }, (response) => {
    if (response) {
      settings = response;
      init();
    }
  });

  function init() {
    if (settings.blockTrackers) {
      blockTrackers();
    }

    if (settings.blockInappropriate) {
      scanForInappropriateContent();
    }

    if (settings.enforceSafeSearch) {
      enforceSafeSearch();
    }

    if (settings.blockAnalytics) {
      blockAnalytics();
    }
  }

  // Block trackers
  function blockTrackers() {
    const trackerDomains = [
      'google-analytics.com',
      'googletagmanager.com',
      'facebook.com/tr',
      'connect.facebook.net',
      'doubleclick.net',
      'scorecardresearch.com',
      'chartbeat.com'
    ];

    // Block scripts from tracker domains
    const scripts = document.querySelectorAll('script[src]');
    scripts.forEach(script => {
      const src = script.getAttribute('src');
      if (src && trackerDomains.some(domain => src.includes(domain))) {
        script.remove();
        notifyBlocked('tracker');
      }
    });

    // Block tracker iframes
    const iframes = document.querySelectorAll('iframe[src]');
    iframes.forEach(iframe => {
      const src = iframe.getAttribute('src');
      if (src && trackerDomains.some(domain => src.includes(domain))) {
        iframe.remove();
        notifyBlocked('tracker');
      }
    });
  }

  // Scan for inappropriate content
  function scanForInappropriateContent() {
    const inappropriateKeywords = [
      'casino', 'gambling', 'bet now', 'adult content',
      'xxx', 'porn', '18+', 'explicit'
    ];

    const textContent = document.body.innerText.toLowerCase();
    
    for (const keyword of inappropriateKeywords) {
      if (textContent.includes(keyword)) {
        // Count occurrences
        const occurrences = (textContent.match(new RegExp(keyword, 'gi')) || []).length;
        
        if (occurrences > 3) {
          // Too many inappropriate keywords, show warning
          showContentWarning();
          notifyBlocked('content');
          break;
        }
      }
    }
  }

  // Enforce SafeSearch on search engines
  function enforceSafeSearch() {
    const url = window.location.href;

    // Google
    if (url.includes('google.com/search')) {
      const urlObj = new URL(url);
      if (!urlObj.searchParams.has('safe') || urlObj.searchParams.get('safe') !== 'active') {
        urlObj.searchParams.set('safe', 'active');
        window.location.href = urlObj.href;
      }
    }

    // Bing
    if (url.includes('bing.com/search')) {
      const urlObj = new URL(url);
      if (!urlObj.searchParams.has('adlt') || urlObj.searchParams.get('adlt') !== 'strict') {
        urlObj.searchParams.set('adlt', 'strict');
        window.location.href = urlObj.href;
      }
    }

    // YouTube - Set restricted mode cookie
    if (url.includes('youtube.com')) {
      document.cookie = 'PREF=f2=8000000; path=/; domain=.youtube.com';
    }
  }

  // Block analytics
  function blockAnalytics() {
    // Override common analytics objects
    window.ga = function() {};
    window.gtag = function() {};
    window._gaq = [];
    window.fbq = function() {};
    window._fbq = function() {};
  }

  // Show content warning overlay
  function showContentWarning() {
    const overlay = document.createElement('div');
    overlay.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(0,0,0,0.95);
      z-index: 999999;
      display: flex;
      align-items: center;
      justify-content: center;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', sans-serif;
    `;

    overlay.innerHTML = `
      <div style="background: white; border-radius: 20px; padding: 40px; max-width: 500px; text-align: center;">
        <div style="font-size: 64px; margin-bottom: 20px;">⚠️</div>
        <h2 style="color: #dc2626; font-size: 28px; margin-bottom: 15px;">Content Blocked</h2>
        <p style="color: #4a5568; font-size: 16px; line-height: 1.6; margin-bottom: 30px;">
          This page has been blocked by CleanFinding because it may contain inappropriate content.
        </p>
        <button id="goBackBtn" style="padding: 14px 28px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 24px; font-size: 16px; font-weight: 600; cursor: pointer;">
          Go Back
        </button>
        <button id="continueBtn" style="padding: 14px 28px; background: #e5e7eb; color: #4a5568; border: none; border-radius: 24px; font-size: 16px; font-weight: 600; cursor: pointer; margin-left: 10px;">
          Continue Anyway
        </button>
      </div>
    `;

    document.body.appendChild(overlay);

    document.getElementById('goBackBtn').addEventListener('click', () => {
      window.history.back();
    });

    document.getElementById('continueBtn').addEventListener('click', () => {
      overlay.remove();
    });
  }

  // Notify background script of blocked content
  function notifyBlocked(type) {
    chrome.runtime.sendMessage({ action: 'blockContent', type: type });
  }

  // Observe DOM changes for dynamically loaded content
  const observer = new MutationObserver(() => {
    if (settings.blockTrackers) {
      blockTrackers();
    }
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true
  });

})();
