// CleanFinding Browser Extension - Background Service Worker

// Default settings
const defaultSettings = {
  blockTrackers: true,
  blockInappropriate: true,
  enforceSafeSearch: true,
  blockAnalytics: true
};

// Initialize extension
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    // Set default settings
    chrome.storage.sync.set(defaultSettings);
    
    // Open welcome page
    chrome.tabs.create({
      url: 'https://cleanfinding.com/extension-download.html?installed=true'
    });
  }
});

// Handle omnibox searches
chrome.omnibox.onInputEntered.addListener((text, disposition) => {
  const searchUrl = `https://cleanfinding.com/search.html?q=${encodeURIComponent(text)}&type=search`;
  
  switch (disposition) {
    case 'currentTab':
      chrome.tabs.update({ url: searchUrl });
      break;
    case 'newForegroundTab':
      chrome.tabs.create({ url: searchUrl });
      break;
    case 'newBackgroundTab':
      chrome.tabs.create({ url: searchUrl, active: false });
      break;
  }
});

// Update omnibox suggestions
chrome.omnibox.onInputChanged.addListener((text, suggest) => {
  const suggestions = [
    {
      content: text,
      description: `Search CleanFinding for: ${text}`
    },
    {
      content: text + ' for kids',
      description: `Search: ${text} for kids`
    },
    {
      content: 'how to ' + text,
      description: `Search: how to ${text}`
    }
  ];
  
  suggest(suggestions);
});

// Context menu for "Search with CleanFinding"
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: 'searchCleanFinding',
    title: 'Search with CleanFinding',
    contexts: ['selection']
  });
});

chrome.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId === 'searchCleanFinding' && info.selectionText) {
    const searchUrl = `https://cleanfinding.com/search.html?q=${encodeURIComponent(info.selectionText)}&type=search`;
    chrome.tabs.create({ url: searchUrl });
  }
});

// Enforce SafeSearch on popular search engines
chrome.webNavigation.onBeforeNavigate.addListener((details) => {
  chrome.storage.sync.get('enforceSafeSearch', (data) => {
    if (data.enforceSafeSearch !== false) {
      const url = new URL(details.url);
      
      // Google SafeSearch
      if (url.hostname.includes('google.com') && url.pathname.includes('/search')) {
        if (!url.searchParams.has('safe') || url.searchParams.get('safe') !== 'active') {
          url.searchParams.set('safe', 'active');
          chrome.tabs.update(details.tabId, { url: url.href });
        }
      }
      
      // Bing SafeSearch
      if (url.hostname.includes('bing.com') && url.pathname.includes('/search')) {
        if (!url.searchParams.has('adlt') || url.searchParams.get('adlt') !== 'strict') {
          url.searchParams.set('adlt', 'strict');
          chrome.tabs.update(details.tabId, { url: url.href });
        }
      }
      
      // YouTube Restricted Mode
      if (url.hostname.includes('youtube.com')) {
        // Note: YouTube restricted mode requires cookies, handled in content script
      }
    }
  });
});

// Track blocked content
let blockedCount = 0;

chrome.declarativeNetRequest.onRuleMatchedDebug.addListener((info) => {
  blockedCount++;
  chrome.storage.local.set({ blockedCount });
});

// Badge to show blocked count
function updateBadge() {
  chrome.storage.local.get('blockedCount', (data) => {
    const count = data.blockedCount || 0;
    if (count > 0) {
      chrome.action.setBadgeText({ text: count.toString() });
      chrome.action.setBadgeBackgroundColor({ color: '#10b981' });
    }
  });
}

// Update badge periodically
setInterval(updateBadge, 5000);

// Handle messages from content scripts
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'blockContent') {
    blockedCount++;
    chrome.storage.local.set({ blockedCount });
    updateBadge();
    sendResponse({ success: true });
  }
  
  if (request.action === 'getSettings') {
    chrome.storage.sync.get(defaultSettings, (settings) => {
      sendResponse(settings);
    });
    return true; // Keep channel open for async response
  }
});

// Reset blocked count daily
chrome.alarms.create('resetBlockedCount', { periodInMinutes: 1440 });

chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'resetBlockedCount') {
    blockedCount = 0;
    chrome.storage.local.set({ blockedCount: 0 });
    chrome.action.setBadgeText({ text: '' });
  }
});

console.log('CleanFinding Extension: Background service worker initialized');
